implementation module stopwatch

//  ********************************************************************************
//  Clean tutorial example program.
//  
//  This program defines a stopwatch process component. 
//  It uses three timers to track the seconds, minutes, and hours separately.
//  Message passing is used to reset, pause, and continue timing. 
//  The current time is displayed using a dialogue.
//  ********************************************************************************

import StdEnv,StdIO

::  NoState
    =   NoState
::  DialogIds
    =   {   secondsId       :: Id
        ,   minutesId       :: Id
        ,   hoursId         :: Id
        ,   dialogId        :: Id
        }
::  TimerInfo
    =   {   timerId         :: Id
        ,   timerRId        :: RId StopwatchCommands
        ,   timerInterval   :: TimerInterval
        }
::  StopwatchCommands
    =   Reset
    |   Pause
    |   Continue
    |   Close

second  :== ticksPerSecond
minute  :== 60*second
hour    :== 60*minute

openDialogIds :: *env -> (DialogIds,*env) | Ids env
openDialogIds env
    #   ([secondsid,minutesid,hoursid,dialogid:_],env) = openIds 4 env
    =   (   {   secondsId=secondsid
            ,   minutesId=minutesid
            ,   hoursId  =hoursid
            ,   dialogId =dialogid
            }
        ,   env
        )

openTimerInfos :: *env -> ([TimerInfo],*env) | Ids env
openTimerInfos env
    #   (tids,env)  = openIds  3 env
    #   (rids,env)  = openRIds 3 env
    #   intervals   = [second,minute,hour]
    =   (   [   {timerId=tid,timerRId=rid,timerInterval=i}
            \\  tid<-tids & rid<-rids & i<-intervals
            ]
        ,   env
        )

stopwatch :: (RId StopwatchCommands) -> ProcessGroup NDIProcess
stopwatch rid
    =   ProcessGroup NoState (NDIProcess NoState [initialise`] [ProcessShareGUI])
where
    initialise` ps
        #   (dialogIds,ps)  = accPIO openDialogIds ps
        #   (timerInfos,ps) = accPIO openTimerInfos ps
        =   initialise rid dialogIds timerInfos ps

initialise :: (RId StopwatchCommands) DialogIds [TimerInfo]
              (PSt .l .p) -> (PSt .l .p)
initialise rid {secondsId,minutesId,hoursId,dialogId} timerinfos ps
    #   (errors,ps)     = seqList [  openTimer 0 (tdef timerinfo)
                                  \\ timerinfo<-timerinfos
                                  ]  ps
    |   any ((<>) NoError) errors
        =   closeProcess ps
    #   (error,ps)      = openDialog NoState ddef ps
    |   error<>NoError
        =   closeProcess ps
    #   (error,ps)      = openReceiver NoState rdef ps
    |   error<>NoError
        =   closeProcess ps
    |   otherwise
        =   ps
where
    tdef {timerId,timerRId,timerInterval}
        =   Timer timerInterval
                (   Receiver timerRId receive []
                )
                [   TimerId         timerId
                ,   TimerFunction   tick
                ]
    where
        tick nrElapsed (time,ps)
            #   time    = (time+nrElapsed) mod (maxunit timerInterval)
            =   (time,setText (textid timerInterval) (toString time) ps)
        
        setText id text ps
            =   appPIO (setWindow dialogId [setControlTexts [(id,text)]]) ps
        
        receive Reset (time,ps)
            #   ps  = appListPIO [disableTimer timerId,enableTimer timerId] ps
            #   ps  = setText (textid timerInterval) "00" ps
            =   (0,ps)
        receive Pause (time,ps)
            =   (time,appPIO (disableTimer timerId) ps)
        receive Continue (time,ps)
            =   (time,appPIO (enableTimer timerId) ps)
            
        textid interval
            |   timerInterval==second
                =   secondsId
            |   timerInterval==minute
                =   minutesId
            |   timerInterval==hour
                =   hoursId
        maxunit interval
            |   timerInterval==second
                =   60
            |   timerInterval==minute
                =   60
            |   timerInterval==hour
                =   24
    
    ddef=   Dialog "Stopwatch"
                (   CompoundControl
                (   ListLS  [  TextControl text [ControlPos (Left,zero)]
                            \\ text<-["Hours:","Minutes:","Seconds:"]
                            ]
                )   []
                :+: CompoundControl
                (   ListLS  [  TextControl "00" [ControlPos (Left,zero)
                                                ,ControlId  id
                                                ]
                            \\ id<-[hoursId,minutesId,secondsId]
                            ]
                )   []
                )
                [   WindowClose (noLS closeProcess)
                ,   WindowId    dialogId
                ]
    rdef
        =   Receiver rid (noLS1 receive) []
    where
        receive Close ps
            =   closeProcess ps
        receive msg ps
            =   snd (seqList [syncSend timerRId msg \\ {timerRId}<-timerinfos] ps)
